/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: $
   Last Modification:
    $Author: Robert $
    $Modtime: $
    $Revision: 6462 $

   Targets:
     MCF51CN128   : yes

   Description:
   cifX parallel DPM Interface

   Changes:

     Version   Date        Author   Description
      ----------------------------------------------------------------------------------
      2        23.07.2014  RM       changed function parameters of MCF51CN128_ParDPM_Read
                                    and MCF51CN128_ParDPM_Write according to new toolkit types
      1        01.03.2011  SS       initial version

**************************************************************************************/

/*****************************************************************************/
/*! \file MCF51CN128ParDPMInterface.c
*   cifX parallel DPM interface function implementation                               */
/*****************************************************************************/
#include "cifXToolkit.h"
#include "MCF51CN128ParDPMInterface.h"

#define SWAP(a) ( (((a) & 0x00FF) << 8) | (((a) & 0xFF00) >> 8) )

static uint16_t s_abHskMirrorCell[16] = { 0x00B0, /* Mirror cell of system channel */
                                          0x00B4, /* Mirror cell of sync bits */
                                          0x0300, /* Mirror cell of communication channel 0 */
                                          0x0000, /* 0x4000 = Mirror cell of communication channel 1 */
                                          0x0000, /* 0x7D00 = Mirror cell of communication channel 2 */
                                          0x0000, /* 0xBA00 = Mirror cell of communication channel 3 */
                                          0x0000, /* application channels are not yet supported */
                                          0x0000, /* application channels are not yet supported */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                          0x0000, /* reserved */
                                       };


#ifdef NX50_PARDPM16
/*****************************************************************************/
/*! Read a number of bytes from 16Bit-Mode DPM interface
*   \param pvDst         Buffer to store read data
*   \param ulDpmAddr     Address in DPM to read data from
*   \param ulLen         Number of bytes to read                             */
/*****************************************************************************/
static void Read_DPM16(void* pvDst, uint32_t ulDpmAddr, uint32_t ulLen)
{
  uint32_t ulIdx = 0;
  
  do
  {
    if ((ulDpmAddr&0x1) || (ulIdx+1 == ulLen))
    {
      uint16_t uiTmp = *(uint16_t*)((ulDpmAddr+ulIdx)&0xFFFFFFFE);
      *((uint8_t*)pvDst+ulIdx) = ((uint8_t*)&uiTmp)[(~(ulDpmAddr+ulIdx))&0x1];
      ulIdx++;
      
    } else
    {
      uint16_t* puiDst = (uint16_t*)((uint8_t*)pvDst+ulIdx);
      *puiDst = SWAP(*(uint16_t*)(ulDpmAddr+ulIdx));
      ulIdx += 2;
    }
  } while(ulLen != ulIdx);
}

/*****************************************************************************/
/*! Write a number of bytes to 16Bit-Mode DPM interface
*   \param ulDpmAddr     Address in DPM to write data
*   \param pvSrc         Buffer storing data to write
*   \param ulLen         Number of bytes to write                            */
/*****************************************************************************/
static void Write_DPM16(uint32_t ulDpmAddr, void* pvSrc, uint32_t ulLen)
{
  uint32_t ulIdx = 0;
  
  do
  {
    if ((ulDpmAddr&0x1) || (ulIdx+1 == ulLen))
    {
      uint16_t uiTmp = *(uint16_t*)((ulDpmAddr+ulIdx)&0xFFFFFFFE);
      if ((ulDpmAddr+ulIdx)&0x1)
        uiTmp = (uiTmp&0x00FF) | (uint16_t)(*((uint8_t*)pvSrc+ulIdx))<<8;
      else
        uiTmp = (uiTmp&0xFF00) | (uint16_t)(*((uint8_t*)pvSrc+ulIdx));
      *(uint16_t*)((ulDpmAddr+ulIdx)&0xFFFFFFFE) = uiTmp;
      ulIdx++;
      
    } else
    {
      uint16_t uiTmp = *((uint16_t*)((uint8_t*)pvSrc+ulIdx));
      *(uint16_t*)((uint8_t*)(ulDpmAddr+ulIdx)) = SWAP(uiTmp);
      ulIdx += 2;
    }
  } while (ulLen != ulIdx);
}

/*****************************************************************************/
/*! Read from handshake cell area, verified with handshake cell mirror  
*   \param ulDpmAddr  Start address of DPM
*   \param pbHskAddr  Address of handshake cell to read                   
*   \param pbDst      Buffer to store handshake cell values
*   \param bLen       Bytes to read from handshake area                  */
/*****************************************************************************/
static void ReadHskArea(uint32_t ulDpmAddr, uint16_t* pusHskAddr, uint16_t* pusDst, uint8_t bLen)
{
  /* Calculate offset of handshake cell in DPM */
  uint8_t  bHskOffset = (uint8_t)((uint32_t)pusHskAddr-ulDpmAddr);
  
  if (bLen ==1)
  {
    /* Get offset of mirrored handshake cell in DPM from lookup table */
    uint16_t usHskMirrorOffset = s_abHskMirrorCell[bHskOffset>>2];
    
    /* Read handshake cell value */
    Read_DPM16(pusDst, (uint32_t)pusHskAddr, 1);
    
    if (usHskMirrorOffset)
    {
      /* Read handshake cell value from mirror */
      uint8_t bHskMirrorVal = 0;

      Read_DPM16(&bHskMirrorVal, ulDpmAddr + usHskMirrorOffset + (bHskOffset&0x3), 1);

      /* Loop until handshake cell and mirror match */
      while (*(uint8_t*)pusDst != bHskMirrorVal)
        Read_DPM16(pusDst, (uint32_t)pusHskAddr, 1);
    }
  } else
  {
    while (bLen)
    {
      /* Get offset of mirrored handshake cell in DPM from lookup table */
      uint16_t usHskMirrorOffset = s_abHskMirrorCell[bHskOffset>>2];
      
      /* Read handshake cell value */
      Read_DPM16(pusDst, (uint32_t)pusHskAddr, 2);
      
      if (usHskMirrorOffset)
      {
        /* Read handshake cell value from mirror */
        uint16_t usHskMirrorVal = 0;
  
        Read_DPM16(&usHskMirrorVal, ulDpmAddr + usHskMirrorOffset + (bHskOffset&0x3), 2);
  
        /* Loop until handshake cell and mirror match */
        while (*pusDst != usHskMirrorVal)
          Read_DPM16(pusDst, (uint32_t)pusHskAddr, 2);
      }
      pusDst++;
      bHskOffset+=2;
      pusHskAddr++;
      bLen-=2;
    }
  }
}

/*****************************************************************************/
/*! Write handshake cell area, verified by subsequent read
*   \param ulDpmAddr     Start address of DPM
*   \param pbHskCellAddr Address of handshake cell to write                   
*   \param pbValue       Buffer holding values to write to handshake area
*   \param bLen          Bytes to write to handshake area                    */
/*****************************************************************************/
static void WriteHskArea(uint32_t ulDpmAddr, uint8_t* pbHskAddr, uint8_t* pbValue, uint8_t bLen)
{
  /* Write handshake cell area */
  Write_DPM16((uint32_t)pbHskAddr, pbValue, bLen);
  
  for (;;)
  {
    char szCookie[5] = {0};
    Read_DPM16(szCookie, ulDpmAddr, 4);
    
    /* Write access to handshake cell is completed, if a subsequent read access succeeds */
    if( (0 == OS_Strcmp( szCookie, CIFX_DPMSIGNATURE_FW_STR)) ||
        (0 == OS_Strcmp( szCookie, CIFX_DPMSIGNATURE_BSL_STR)) )
      break;
  }
}
#else

/*****************************************************************************/
/*! Read from handshake cell area, verified with handshake cell mirror  
*   \param ulDpmAddr  Start address of DPM
*   \param pbHskAddr  Address of handshake cell to read                   
*   \param pbDst      Buffer to store handshake cell values
*   \param bLen       Bytes to read from handshake area                      */
/*****************************************************************************/
static void ReadHskArea(uint32_t ulDpmAddr, uint8_t* pbHskAddr, uint8_t* pbDst, uint8_t bLen)
{
  /* Calculate offset of handshake cell in DPM */
  uint8_t  bHskOffset = (uint8_t)((uint32_t)pbHskAddr-ulDpmAddr);
  
  while (bLen--)
  {
    /* Get offset of mirrored handshake cell in DPM from lookup table */
    uint16_t usHskMirrorOffset = s_abHskMirrorCell[bHskOffset>>2];
    /* Read handshake cell value */
    *pbDst                     = *(volatile uint8_t*)pbHskAddr;
    
    if (usHskMirrorOffset)
    {
      /* Read handshake cell value from mirror */
      uint8_t bHskMirrorVal = *(uint8_t*)(ulDpmAddr         + 
                                          usHskMirrorOffset + 
                                          (bHskOffset&0x3));
      /* Loop until handshake cell and mirror match */
      while (*pbDst != bHskMirrorVal)
        *pbDst = *(volatile uint8_t*)pbHskAddr;
    }
    pbDst++;
    bHskOffset++;
    pbHskAddr++;
  }
}

/*****************************************************************************/
/*! Write handshake cell area, verified by subsequent read
*   \param ulDpmAddr     Start address of DPM
*   \param pbHskCellAddr Address of handshake cell to write                   
*   \param pbValue       Buffer holding values to write to handshake area
*   \param bLen          Bytes to write to handshake area                    */
/*****************************************************************************/
static void WriteHskArea(uint32_t ulDpmAddr, uint8_t* pbHskAddr, uint8_t* pbValue, uint8_t bLen)
{
  while (bLen--)
  {
    /* Write handshake cell area */
    *pbHskAddr++ = *pbValue++;
    for (;;)
    {
      /* Write access to handshake cell is completed, if a subsequent read access succeeds */
      if( (0 == OS_Strcmp( (char*)ulDpmAddr, CIFX_DPMSIGNATURE_FW_STR)) ||
          (0 == OS_Strcmp( (char*)ulDpmAddr, CIFX_DPMSIGNATURE_BSL_STR)) )
        break;
    }
  }
}
#endif /* NX50_PARDPM16 */

/*****************************************************************************/
/*! Init DPM interface
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t MCF51CN128_ParDPM_Init( void)
{ 
  return 0;
}

/*****************************************************************************/
/*! Write a number of bytes to DPM interface
*   \param pvDevInstance Toolkit device instance
*   \param pvAddr        Address in DPM to write data to
*   \param pvData        Buffer holding write data
*   \param ulLen         Number of bytes to write
*   \return Pointer to DPM                                                   */
/*****************************************************************************/
void* MCF51CN128_ParDPM_Write( void* pvDevInstance, void* pvAddr, void* pvData, uint32_t ulLen)
{ 
  PDEVICEINSTANCE ptDevInst     = (PDEVICEINSTANCE)pvDevInstance;
  uint32_t        ulDPMOffset   = ((uint32_t)pvAddr) - (uint32_t)ptDevInst->pbDPM;
  
  if (NETX_SYSTEM_CHANNEL_SIZE == (ulDPMOffset&(~(NETX_HANDSHAKE_CHANNEL_SIZE-1)))) 
  {
    WriteHskArea((uint32_t)ptDevInst->pbDPM, pvAddr, pvData, (uint8_t)ulLen);
      
  } else
  {
    #ifdef NX50_PARDPM16
      Write_DPM16( (uint32_t)pvAddr, pvData, ulLen);
    #else
      OS_Memcpy( pvAddr, pvData, ulLen);
    #endif /* NX50_PARDPM_16BIT */
  }

  return pvAddr;
}

/*****************************************************************************/
/*! Read a number of bytes from DPM interface
*   \param pvDevInstance Toolkit device instance
*   \param pvAddr        Address in DPM to read data from
*   \param pvData        Buffer to store read data
*   \param ulLen         Number of bytes to read
*   \return Pointer to read buffer                                                   */
/*****************************************************************************/
void* MCF51CN128_ParDPM_Read( void* pvDevInstance, void* pvAddr, void* pvData, uint32_t ulLen)
{
  PDEVICEINSTANCE ptDevInst       = (PDEVICEINSTANCE)pvDevInstance;
  uint32_t        ulDPMOffset     = ((uint32_t)pvAddr) - (uint32_t)ptDevInst->pbDPM;
    
  if (NETX_SYSTEM_CHANNEL_SIZE == (ulDPMOffset&~(NETX_HANDSHAKE_CHANNEL_SIZE-1))) 
  {
    ReadHskArea((uint32_t)ptDevInst->pbDPM, pvAddr, pvData, (uint8_t)ulLen);
    
  } else
  {
    #ifdef NX50_PARDPM16
      Read_DPM16( pvData, pvAddr, ulLen);
    #else
      OS_Memcpy( pvData, pvAddr, ulLen);
    #endif /* NX50_PARDPM16 */
  }
  
  return pvData;
}
